// Expected global variables:
/*global domHelper heapHelper mainBrowser onErrorHandler remoteHelpers toolUI */

// Allow 'eval' for debugging code execution and leading underscores for invoke functions
/*jshint evil: true, nomen: false*/

var messageHandlers;

var remoteCode = {

    // The queue of notification messages that have been received before the console is ready
    consoleNotificationQueue: [],

    initialize: function () {
        /// <summary>
        ///     Gets the remote script code ready for use. 
        ///     Sets up communication with the VS side dom explorer
        /// </summary>

        // Listen for the navigate events
        mainBrowser.addEventListener("BeforeScriptExecute", remoteCode.onBeforeScriptExecute);

        if (remoteCode.port) {
            // Remove the previous port
            remoteCode.port.removeEventListener("message");
            remoteCode.portReady = false;
        }

        // Create a port
        remoteHelpers.initialize("MemoryAnalyzerPort", remoteCode.initializePage, "__VISUALSTUDIO_MEMORYANALYZER_ADDED", remoteCode.onDetach);
    },

    initializePage: function () {
        /// <summary>
        ///     This method initializes the Dom Explorer by injecting the F12 handler
        ///     and informing the VS side that it needs to repopulate the tree view
        ///     It sends the message over the communication port
        /// </summary>

        try {
            remoteCode.addRemotePageFunctions(mainBrowser.document.parentWindow);
            remoteCode.initializeConsoles(mainBrowser);

            var connectionInfo = { docMode: mainBrowser.document.documentMode, contextInfo: mainBrowser.document.parentWindow.location.href };
            remoteHelpers.port.postMessage("Handshake:" + JSON.stringify(connectionInfo));

            // Reset the document timeout
            remoteHelpers.initializeDocumentTries = 0;
        } catch (e) {
            remoteCode.onDocumentNotReady();
        }
    },

    onDocumentNotReady: function () {
        /// <summary>
        ///     This method should be called when the document is not yet ready for diagnostics to attach and populate.
        ///     It will increment the timeout counter and post back to the VS side asking for it to re-try the initialize call.
        /// </summary>

        if (remoteHelpers.initializeDocumentTries < remoteHelpers.initializeDocumentMaxTries) {
            remoteHelpers.initializeDocumentTries++;
            remoteHelpers.port.postMessage("DocumentNotYetReady");
        } else {
            onErrorHandler("Document timed out", "remote.js", 0);
        }
    },

    initializeConsoles: function (windowToAttach) {
        /// <summary>
        ///     Recurses through all the frames on the window and calls 
        ///     'initializeConsole' to initialize the console on that frame
        /// </summary>
        /// <param name="windowToAttach" type="Object">
        ///     The IDispatch window that we should inspect for frames
        /// </param>

        var realWindow = null;

        try {
            // Try to get the window object that javascript expects
            realWindow = windowToAttach.document.parentWindow;
        } catch (ex) {
            // Ignore this beforeScriptExecute, as the window is not valid and cannot attach a console
            return;
        }

        // Attach a console to this window
        remoteCode.initializeConsole(realWindow);

        // Use recursion to add a console to each frame
        if (realWindow.frames) {
            for (var i = 0; i < realWindow.frames.length; i++) {
                var frame = realWindow.frames[i];
                if (frame) {
                    var iframe = domHelper.getCrossSiteWindow(realWindow, frame);
                    remoteCode.initializeConsoles(iframe);
                }
            }
        }
    },

    initializeConsole: function (realWindow) {
        /// <summary>
        ///     Initializes the console for the specified window
        /// </summary>
        /// <param name="realWindow" type="Object">
        ///     The window object that we add the console to
        /// </param>

        var modifiedConsole = false;
        var consoleObj = realWindow.console;
        var injectedConsole = {};

        if (!consoleObj) {
            // Create a new console object
            consoleObj = realWindow.console = {};
            modifiedConsole = true;
        }

        if (!consoleObj.takeHeapSnapshot) {
            consoleObj.takeHeapSnapshot = function (msg) {
                remoteCode.onConsoleExecute("takeHeapSnapshot", msg);
            };
            injectedConsole.takeHeapSnapshot = consoleObj.takeHeapSnapshot;
            modifiedConsole = true;
        }

        if (!consoleObj.profileMark) {
            consoleObj.profileMark = function (msg) {
                if (msg && msg.length > 0) {
                    realWindow.msWriteProfilerMark("__VISUALSTUDIO_MEMORYANALYZER_ " + msg);
                } else {
                    realWindow.msWriteProfilerMark("__VISUALSTUDIO_MEMORYANALYZER_");
                }
            };
            injectedConsole.profileMark = consoleObj.profileMark;
            modifiedConsole = true;
        }

        if (modifiedConsole) {
            // Add the detach handler to remove the console object when we detach
            var detachHandler = function () {
                remoteCode.onDetach(realWindow, injectedConsole);
                toolUI.removeEventListener("detach", detachHandler);
            };

            // Add the unload handler to remove the detach handler on a navigate,
            // The actual page navigation will take care of cleaning up the consoles.
            // This enables us to still catch any console function calls during the unload event.
            if (realWindow.addEventListener) {
                // DocumentMode 9 and above
                realWindow.addEventListener("unload", function () {
                    toolUI.removeEventListener("detach", detachHandler);
                });
            } else {
                // Below DocumentMode 9
                realWindow.attachEvent("onunload", function () {
                    toolUI.removeEventListener("detach", detachHandler);
                });
            }

            toolUI.addEventListener("detach", detachHandler);
        }
    },

    onConsoleExecute: function (functionId, data) {
        /// <summary>
        ///     This method is called by functions attached to the window.console
        ///     It processes the function call and communicates information back to the VS side
        /// </summary>
        /// <param name="functionId" type="String">
        ///     The type of the message
        ///     Either a notification type from consoleUtils.consoleNotifyType or a built in function
        ///     such as "cd", "clear", "dir", etc
        /// </param>
        /// <param name="data" type="Object" optional="true">
        ///     The information used for this function call
        /// </param>


        // Allfunctions are notification types
        if (!remoteCode.notifyCallback) {
            // The port is not ready yet, so queue up the notifications
            remoteCode.consoleNotificationQueue.push({ functionId: functionId, data: data });
        } else {
            // Fire the notification directly
            remoteCode.notifyCallback({ notifyType: functionId, message: data });
        }
    },

    onDetach: function () {
        /// <summary>
        ///     This method is called when debugging is detached, so we can perform clean up
        /// </summary>
        /// <disable>JS2078.DoNotDeleteObjectProperties</disable>

        // Reset the document timeout
        remoteHelpers.initializeDocumentTries = 0;

        try {
            if (mainBrowser.document.parentWindow.__VISUALSTUDIO_MEMORYANALYZER_ADDED) {
                // Remove the "we've attached" notification.
                delete mainBrowser.document.parentWindow.__VISUALSTUDIO_MEMORYANALYZER_ADDED;
            }

        } catch (ex) {
            // We should fail gracefully if there are access issues
        }
    },

    onBeforeScriptExecute: function (dispatchWindow) {
        /// <summary>
        ///     This method is called back when the main browser is about to execute script, 
        ///     so we should refresh the dom explorer
        /// </summary>
        /// <param name="dispatchWindow" type="Object">
        ///     The IDispatch window that triggered the BeforeScriptExecute event
        /// </param>

        var realWindow = null;

        try {
            // Try to get the window object that javascript expects
            realWindow = dispatchWindow.document.parentWindow;
        } catch (ex) {
            // Ignore this beforeScriptExecute, as the window is not valid and cannot be the root frame
            return;
        }

        // Ensure the new window is the top level one and not a sub frame
        if (realWindow === mainBrowser.document.parentWindow && remoteHelpers.port) {

            // Finish posting any messages from the previous page
            remoteHelpers.postAllMessages();

            remoteCode.initializePage();
        }
    },

    addRemotePageFunctions: function (realWindow) {
        realWindow.__VISUALSTUDIO_MEMORYANALYZER_ADDED = true;
    }
};

var extensionsHelper = {
    __memAnalyzerExtensionInstance: null,

    getMemAnalyzerExtension: function () {
        try {
            if (!extensionsHelper.__memAnalyzerExtensionInstance) {
                extensionsHelper.__memAnalyzerExtensionInstance = extensions.loadExtension("MemAnalyzer.dll", "{5B8BEAA5-1131-4C42-9B48-D7F69E579A8A}");
            }
            return extensionsHelper.__memAnalyzerExtensionInstance;
        } catch (e) {
            return null;
        }
    }
};

messageHandlers = {
    /// <summary>
    ///     Object that acts as the message handler for messages that get sent from the VS side
    ///     The messages contain a command property that corrisponds to a function on this 
    ///     object that processes that message
    /// </summary>

    createSnapshot: function (screenshotTargetWidth, screenshotTargetHeight, screenshotKeepAspectRatio) {
        /// <summary>
        ///     Gets a complete set of snapshot information including the heap, private bytes, and screenshot.
        /// </summary>
        /// <param name="screenshotTargetWidth" type="Number">
        ///     The requested width of the created image (Must be > 0).
        /// </param>
        /// <param name="screenshotTargetHeight" type="Number">
        ///     The requested height of the created image (Must be > 0).
        /// </param>
        /// <param name="screenshotKeepAspectRatio" type="Boolean">
        ///     True to maintain the existing aspect ratio of the screen when stretching to the
        ///     target sizes. False to just use the target sizes.
        /// </param>
        /// <returns type="String">
        ///     The full path to the snapshot file or null.
        /// </returns>

        var memAnalyzerExtension = extensionsHelper.getMemAnalyzerExtension();

        if (!memAnalyzerExtension) {
            return null;
        }

        var snapshotPath = "";
        try {
            snapshotPath = memAnalyzerExtension.createSnapshot(screenshotTargetWidth, screenshotTargetHeight, screenshotKeepAspectRatio);
        } catch (ex) {
            if (ex.number && !ex.message && !ex.description) {
                ex.message = memAnalyzerExtension.getHresultMessage(ex.number);
            }

            return { error: ex };
        }

        return { filePath: snapshotPath };
    },

    deleteSnapshot: function () {
        /// <summary>
        ///     Deletes a snapshot file.
        /// </summary>

        var memAnalyzerExtension = extensionsHelper.getMemAnalyzerExtension();

        if (memAnalyzerExtension) {
            memAnalyzerExtension.deleteSnapshot();
        }
    },

    getTargetProcessId: function () {
        /// <summary>
        ///     Gets the process id of the diagnostics target.
        /// </summary>
        /// <returns type="Number">
        ///     The process id.
        /// </returns>

        var memAnalyzerExtension = extensionsHelper.getMemAnalyzerExtension();

        if (!memAnalyzerExtension) {
            return 0;
        }

        return memAnalyzerExtension.getProcessId();
    },

    registerConsoleCallbacks: function (notifyCallback) {
        /// <summary>
        ///     Stores the callbacks used to send messages and commands to the VS side
        /// </summary>
        /// <param name="notifyCallback" type="Function">
        ///     The callback function that is used for sending info, warnings, errors, and assert
        ///     to the VS side console
        /// </param>  


        // Store the callback for console notification messages
        remoteCode.notifyCallback = notifyCallback;

        // If we queued up any messages while the port was being created, then fire them now
        for (var i = 0; i < remoteCode.consoleNotificationQueue.length; i++) {
            var notification = remoteCode.consoleNotificationQueue[i];
            remoteCode.onConsoleExecute(notification.functionId, notification.data);
        }
    },
};

remoteCode.initialize();

// SIG // Begin signature block
// SIG // MIIaiQYJKoZIhvcNAQcCoIIaejCCGnYCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFH7ZbHwhT3Wi
// SIG // SKI248WfJ3cib6zwoIIVeTCCBLowggOioAMCAQICCmEC
// SIG // kkoAAAAAACAwDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OVoXDTEz
// SIG // MDQwOTIyMjU1OVowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpCOEVDLTMwQTQtNzE0NDElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAM1jw/ei
// SIG // tUfZ+TmUU6xrj6Z5OCH00W49FTgWwXMsmY/74Dxb4aJM
// SIG // i7Kri7TySse5k1DRJvWHU7B6dfNHDxcrZyxk62DnSozg
// SIG // i17EVmk3OioEXRcByL+pt9PJq6ORqIHjPy232OTEeAB5
// SIG // Oc/9x2TiIxJ4ngx2J0mPmqwOdOMGVVVJyO2hfHBFYX6y
// SIG // cRYe4cFBudLSMulSJPM2UATX3W88SdUL1HZA/GVlE36V
// SIG // UTrV/7iap1drSxXlN1gf3AANxa7q34FH+fBSrubPWqzg
// SIG // FEqmcZSA+v2wIzBg6YNgrA4kHv8R8uelVWKV7p9/ninW
// SIG // zUsKdoPwQwTfBkkg8lNaRLBRejkCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBTNGaxhTZRnK/avlHVZ2/BYAIOhOjAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBRHNbfNh3cgLwCp8aZ3xbI
// SIG // kAZpFZoyufNkENKK82IpG3mPymCps13E5BYtNYxEm/H0
// SIG // XGGkQa6ai7pQ0Wp5arNijJ1NUVALqY7Uv6IQwEfVTnVS
// SIG // iR4/lmqPLkAUBnLuP3BZkl2F7YOZ+oKEnuQDASETqyfW
// SIG // zHFJ5dod/288CU7VjWboDMl/7jEUAjdfe2nsiT5FfyVE
// SIG // 5x8a1sUaw0rk4fGEmOdP+amYpxhG7IRs7KkDCv18elId
// SIG // nGukqA+YkqSSeFwreON9ssfZtnB931tzU7+q1GZQS/DJ
// SIG // O5WF5cFKZZ0lWFC7IFSReTobB1xqVyivMcef58Md7kf9
// SIG // J9d/z3TcZcU/MIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggR8MIIEeAIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIGeMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBTPoup4UdAWT7w11eWOATYV
// SIG // 0kD6GzA+BgorBgEEAYI3AgEMMTAwLqAUgBIAcgBlAG0A
// SIG // bwB0AGUALgBqAHOhFoAUaHR0cDovL21pY3Jvc29mdC5j
// SIG // b20wDQYJKoZIhvcNAQEBBQAEggEAN+uLXxgn4zZB+OMr
// SIG // wxB4o4U29+pgvpftJ/ipI+exQakH83FGaSPr80928/HN
// SIG // p6QKzllRaMXjqjLOPdsPsC1OuWf9HOLWvOYTEaW6nx/l
// SIG // km5sRS2BiZnePN/+qV7CuJxzxX8/lno+PcQ/kNdiVsEr
// SIG // ijY8Cu9STk7PV8poBinTSJYv6j9OTLNrZ8BuJawtOca7
// SIG // +flzlEfDvGnRbdq0IcTKAj6fAVAXspmPAxGyh9NZZwyx
// SIG // ufnbfOCiUWlBDvrQtfQk339lWvKrpuM+EGNd8kTCIaof
// SIG // ACPCE/xiiTpd2QRE7s4+vUFrMPidydvZltbxIJmWdeD1
// SIG // X33/gMUoPmr8hNeAD6GCAh8wggIbBgkqhkiG9w0BCQYx
// SIG // ggIMMIICCAIBATCBhTB3MQswCQYDVQQGEwJVUzETMBEG
// SIG // A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
// SIG // ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9u
// SIG // MSEwHwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQ
// SIG // Q0ECCmECkkoAAAAAACAwCQYFKw4DAhoFAKBdMBgGCSqG
// SIG // SIb3DQEJAzELBgkqhkiG9w0BBwEwHAYJKoZIhvcNAQkF
// SIG // MQ8XDTEzMDMxNTA2MzM1NVowIwYJKoZIhvcNAQkEMRYE
// SIG // FJfL0h0bhBYVfg8GbgsufXLuuzhfMA0GCSqGSIb3DQEB
// SIG // BQUABIIBADgqFcOOpq4q2S1QVICQ3lhenw0KKCUpUXQO
// SIG // Kb//MR5JbFxbBKiJb3x8Kh0tROynOw4Biq+RkPwp7HUI
// SIG // 0/OAYQ08XoUvRdHGnunR+nGf5SeSPWvhsKhO9RBtxqO+
// SIG // IwlwPMDyiXAHGRDHiZ1AnN1N3EcQsLTzAlEwbtjGRCcL
// SIG // 5Uwe1zEurc3WNku3FAWt9wuajzno5dXvx0uAWuEIOhVl
// SIG // NvxGZi7DOQNo/tyllvJHN2aN36TAdp1dTBNM7cj8ypPG
// SIG // pHMZd6dQWQvsFHa+atg/+EAaF5h0sjsphrz11kFcF8ZV
// SIG // fneiwkW4ENTJqFXB1hnTgflBsJUswgDtnsC9xVkoFyo=
// SIG // End signature block
